<<?php
//checkout
require 'db.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// flash helper (same style as in cart)
function set_flash(string $message, string $type = 'success'): void {
    $_SESSION['flash'] = [
        'message' => $message,
        'type'    => $type,
    ];
}

// Ensure cart
$cart = $_SESSION['cart'] ?? [];
if (empty($cart)) {
    set_flash('Your cart is empty. Add items before checking out.', 'error');
    header("Location: /cart");
    exit;
}

// Fetch products in cart
$ids = array_keys($cart);
$placeholders = implode(',', array_fill(0, count($ids), '?'));
$stmt = $pdo->prepare("SELECT * FROM products WHERE id IN ($placeholders) AND is_active = 1");
$stmt->execute($ids);
$rows = $stmt->fetchAll();

// Map
$productMap = [];
foreach ($rows as $row) {
    $productMap[$row['id']] = $row;
}

// Recalculate total from DB
$orderTotal = 0;
$cartItems = [];
foreach ($cart as $pid => $qty) {
    if (!isset($productMap[$pid])) continue;
    $p   = $productMap[$pid];
    $qty = max(1, (int)$qty);
    $line = $p['price'] * $qty;
    $orderTotal += $line;
    $cartItems[$pid] = $qty;
}

if ($orderTotal <= 0 || empty($cartItems)) {
    set_flash('Your cart is empty or invalid. Please add items again.', 'error');
    header("Location: /cart");
    exit;
}

// Form handling
$errors = [];
$orderCreated = false;
$orderId = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name    = trim($_POST['name'] ?? '');
    $phone   = trim($_POST['phone'] ?? '');
    $email   = trim($_POST['email'] ?? '');
    $town    = trim($_POST['town'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $notes   = trim($_POST['notes'] ?? '');

    if ($name === '') {
        $errors[] = "Name is required.";
    }
    if ($phone === '') {
        $errors[] = "Phone number is required.";
    }
    if ($address === '') {
        $errors[] = "Delivery address is required.";
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Insert order
            $orderStmt = $pdo->prepare("
                INSERT INTO orders
                    (customer_name, phone, email, town, address, notes, total_amount, status, payment_status)
                VALUES
                    (:customer_name, :phone, :email, :town, :address, :notes, :total_amount, :status, :payment_status)
            ");

            $orderStmt->execute([
                ':customer_name'   => $name,
                ':phone'           => $phone,
                ':email'           => $email ?: null,
                ':town'            => $town ?: null,
                ':address'         => $address,
                ':notes'           => $notes ?: null,
                ':total_amount'    => $orderTotal,
                ':status'          => 'pending',
                ':payment_status'  => 'unpaid',
            ]);

            $orderId = (int)$pdo->lastInsertId();

            // Insert order items
            $itemStmt = $pdo->prepare("
                INSERT INTO order_items
                    (order_id, product_id, product_name, unit_price, quantity, line_total)
                VALUES
                    (:order_id, :product_id, :product_name, :unit_price, :quantity, :line_total)
            ");

            foreach ($cartItems as $pid => $qty) {
                $p = $productMap[$pid];
                $unit = (float)$p['price'];
                $qty  = (int)$qty;
                $line = $unit * $qty;

                $itemStmt->execute([
                    ':order_id'     => $orderId,
                    ':product_id'   => $p['id'],
                    ':product_name' => $p['name'],
                    ':unit_price'   => $unit,
                    ':quantity'     => $qty,
                    ':line_total'   => $line,
                ]);
            }

            $pdo->commit();

            // Optional email notification to admin
            $configPath = __DIR__ . '/admin/config.php';
            if (file_exists($configPath)) {
                require_once $configPath;

                if (defined('SEND_ORDER_EMAIL') && SEND_ORDER_EMAIL && defined('ADMIN_ORDER_EMAIL')) {
                    $subject = "New Order #{$orderId} – CompNet Shop";

                    $bodyLines = [];
                    $bodyLines[] = "New order received on CompNet Shop.";
                    $bodyLines[] = "";
                    $bodyLines[] = "Order ID: #{$orderId}";
                    $bodyLines[] = "Date: " . date('Y-m-d H:i:s');
                    $bodyLines[] = "";
                    $bodyLines[] = "Customer: $name";
                    $bodyLines[] = "Phone: $phone";
                    if ($email) $bodyLines[] = "Email: $email";
                    if ($town)  $bodyLines[] = "Town: $town";
                    $bodyLines[] = "Address: $address";
                    if ($notes) {
                        $bodyLines[] = "";
                        $bodyLines[] = "Notes: $notes";
                    }
                    $bodyLines[] = "";
                    $bodyLines[] = "Total: KSh " . number_format($orderTotal);
                    $bodyLines[] = "";
                    $bodyLines[] = "Items:";
                    foreach ($cartItems as $pid => $qty) {
                        $p = $productMap[$pid];
                        $line = $p['price'] * $qty;
                        $bodyLines[] = "- {$p['name']} x{$qty} (KSh " . number_format($line) . ")";
                    }

                    $body = implode("\n", $bodyLines);

                    $fromDomain = $_SERVER['SERVER_NAME'] ?? 'localhost';
                    $headers = "From: CompNet Shop <no-reply@$fromDomain>\r\n";

                    @mail(ADMIN_ORDER_EMAIL, $subject, $body, $headers);
                }
            }

            // Clear cart
            $_SESSION['cart'] = [];
            $orderCreated = true;

        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Failed to place order. Please try again.";
        }
    }
}

$pageTitle = "Checkout – CompNet";
require 'partials/header.php';
?>

<h1>Checkout</h1>

<?php if ($orderCreated && $orderId): ?>
    <div class="product-panel" style="max-width:640px;">
        <h2>Thank you!</h2>
        <p style="margin-top:8px;">
            Your order has been placed successfully.
        </p>
        <p>
            <strong>Order ID:</strong> #<?= $orderId ?><br>
            <strong>Total:</strong> KSh <?= number_format($orderTotal) ?>
        </p>
        <p style="margin-top:12px;">
            Our team will contact you shortly to confirm delivery and payment.
        </p>
        <p style="margin-top:16px;">
            <a href="/" class="btn-primary" style="text-decoration:none;">Back to Home</a>
        </p>
    </div>
<?php else: ?>

    <?php if (!empty($errors)): ?>
        <div class="flash flash-error">
            <?= htmlspecialchars(implode(' ', $errors)) ?>
        </div>
    <?php endif; ?>

    <div class="checkout-layout">
        <div class="checkout-form-panel">
            <h2>Delivery Details</h2>
            <form method="post" style="display:flex;flex-direction:column;gap:10px;margin-top:10px;">
                <div>
                    <label>Name *</label>
                    <input type="text" name="name" required
                           value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
                </div>
                <div>
                    <label>Phone *</label>
                    <input type="text" name="phone" required
                           value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>">
                </div>
                <div>
                    <label>Email</label>
                    <input type="email" name="email"
                           value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
                </div>
                <div>
                    <label>Town</label>
                    <input type="text" name="town"
                           value="<?= htmlspecialchars($_POST['town'] ?? '') ?>">
                </div>
                <div>
                    <label>Delivery Address *</label>
                    <textarea name="address" rows="3" required><?= htmlspecialchars($_POST['address'] ?? '') ?></textarea>
                </div>
                <div>
                    <label>Notes (optional)</label>
                    <textarea name="notes" rows="3"><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                </div>

                <div style="display:flex;justify-content:space-between;align-items:center;margin-top:6px;">
                    <a href="/cart" class="admin-btn-secondary" style="text-decoration:none;">← Back to cart</a>
                    <button type="submit" class="admin-btn">Place Order</button>
                </div>
            </form>
        </div>

        <div class="checkout-summary-panel">
            <h2>Order Summary</h2>
            <div style="margin-top:8px;">
                <?php foreach ($cartItems as $pid => $qty): ?>
                    <?php $p = $productMap[$pid]; ?>
                    <div class="checkout-summary-item">
                        <div>
                            <div style="font-size:14px;font-weight:600;">
                                <?= htmlspecialchars($p['name']) ?>
                            </div>
                            <div class="muted" style="font-size:12px;">
                                Qty: <?= (int)$qty ?>
                            </div>
                        </div>
                        <div style="text-align:right;font-size:14px;font-weight:600;">
                            KSh <?= number_format($p['price'] * $qty) ?>
                        </div>
                    </div>
                <?php endforeach; ?>

                <div class="checkout-summary-total">
                    <span>Total</span>
                    <strong>KSh <?= number_format($orderTotal) ?></strong>
                </div>

                <p class="muted" style="font-size:12px;margin-top:6px;">
                    Payment options: Pay on delivery in Nairobi or pay via M-PESA / bank for upcountry deliveries.
                </p>
            </div>
        </div>
    </div>

<?php endif; ?>

<?php require 'partials/footer.php'; ?>
