<?php
//cart
require 'db.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Simple flash helper
function set_flash(string $message, string $type = 'success'): void {
    $_SESSION['flash'] = [
        'message' => $message,
        'type'    => $type,
    ];
}

// Ensure cart array exists
if (!isset($_SESSION['cart']) || !is_array($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

$action = $_GET['action'] ?? '';

// Handle cart actions
if ($action === 'add') {
    $id  = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    $qty = isset($_GET['qty']) ? (int)$_GET['qty'] : 1;
    $qty = max(1, $qty);

    if ($id > 0) {
        // Validate product exists and is active
        $stmt = $pdo->prepare("SELECT id FROM products WHERE id = :id AND is_active = 1");
        $stmt->execute([':id' => $id]);
        $exists = $stmt->fetchColumn();

        if ($exists) {
            if (!isset($_SESSION['cart'][$id])) {
                $_SESSION['cart'][$id] = 0;
            }
            $_SESSION['cart'][$id] += $qty;
            set_flash('Item added to cart.');
        } else {
            set_flash('Product not found or inactive.', 'error');
        }
    }

    $return = $_GET['return'] ?? '';
    if ($return) {
        header("Location: $return");
    } else {
        header("Location: /cart");
    }
    exit;
}

if ($action === 'remove') {
    $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    if ($id > 0 && isset($_SESSION['cart'][$id])) {
        unset($_SESSION['cart'][$id]);
        set_flash('Item removed from cart.');
    }
    header("Location: /cart");
    exit;
}

if ($action === 'clear') {
    $_SESSION['cart'] = [];
    set_flash('Cart cleared.');
    header("Location: /cart");
    exit;
}

if ($action === 'update' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $qtyData = $_POST['qty'] ?? [];
    if (is_array($qtyData)) {
        foreach ($qtyData as $pid => $qty) {
            $pid = (int)$pid;
            $qty = (int)$qty;
            if ($pid <= 0) {
                continue;
            }
            if ($qty <= 0) {
                unset($_SESSION['cart'][$pid]);
            } else {
                $_SESSION['cart'][$pid] = $qty;
            }
        }
    }
    set_flash('Cart updated.');
    header("Location: /cart");
    exit;
}

// Build cart view
$cart = $_SESSION['cart'];
$products = [];
$grandTotal = 0;

if (!empty($cart)) {
    $ids = array_keys($cart);
    $placeholders = implode(',', array_fill(0, count($ids), '?'));
    $stmt = $pdo->prepare("
        SELECT *
        FROM products
        WHERE id IN ($placeholders)
    ");
    $stmt->execute($ids);
    $rows = $stmt->fetchAll();

    // Map by id
    $products = [];
    foreach ($rows as $row) {
        $products[$row['id']] = $row;
    }

    // Calculate total
    foreach ($cart as $pid => $qty) {
        if (!isset($products[$pid])) continue;
        $grandTotal += $products[$pid]['price'] * $qty;
    }
}

$pageTitle = "Your Cart – CompNet";
require 'partials/header.php';
?>

<h1>Your Cart</h1>

<?php if (empty($cart) || empty($products)): ?>
    <p class="muted">Your cart is empty.</p>
    <p>
        <a href="/" class="btn-primary" style="text-decoration:none;">Continue Shopping</a>
    </p>
<?php else: ?>
    <form action="/cart.php?action=update" method="post" class="cart-form">
        <div class="table-box">
            <table class="cart-table">
                <thead>
                    <tr>
                        <th style="text-align:left;">Product</th>
                        <th style="text-align:right;">Price</th>
                        <th style="text-align:center;">Qty</th>
                        <th style="text-align:right;">Total</th>
                        <th style="text-align:center;">Remove</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($cart as $pid => $qty): ?>
                        <?php if (!isset($products[$pid])) continue; ?>
                        <?php
                        $p = $products[$pid];
                        $lineTotal = $p['price'] * $qty;
                        ?>
                        <tr>
                            <td>
                                <a href="/product/<?= $p['id'] ?>">
                                    <?= htmlspecialchars($p['name']) ?>
                                </a>
                            </td>
                            <td style="text-align:right;">KSh <?= number_format($p['price']) ?></td>
                            <td style="text-align:center;">
                                <input type="number" name="qty[<?= $p['id'] ?>]"
                                       value="<?= (int)$qty ?>"
                                       min="1"
                                       style="width:70px;padding:4px 6px;border-radius:6px;border:1px solid #ddd;">
                            </td>
                            <td style="text-align:right;">KSh <?= number_format($lineTotal) ?></td>
                            <td style="text-align:center;">
                                <a href="/cart.php?action=remove&id=<?= $p['id'] ?>"
                                   style="color:#b91c1c;font-size:13px;">Remove</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <tr>
                        <th colspan="3" style="text-align:right;">Grand Total</th>
                        <th style="text-align:right;">KSh <?= number_format($grandTotal) ?></th>
                        <th></th>
                    </tr>
                </tfoot>
            </table>
        </div>

        <div class="cart-actions">
            <div>
                <a href="/" class="admin-btn-secondary" style="text-decoration:none;">← Continue Shopping</a>
            </div>
            <div style="display:flex;gap:8px;">
                <a href="/cart.php?action=clear" class="admin-btn-secondary" style="text-decoration:none;">Clear Cart</a>
                <button type="submit" class="admin-btn-secondary">Update Cart</button>
                <a href="/checkout" class="admin-btn" style="text-decoration:none;">Proceed to Checkout</a>
            </div>
        </div>
    </form>
<?php endif; ?>

<?php require 'partials/footer.php'; ?>
