<?php
// admin/order_view.php  with mpesa validation
$pageTitle = "Order Details – CompNet Admin";
require __DIR__ . '/../db.php';
require __DIR__ . '/header.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) {
    echo "<h1>Invalid order ID.</h1>";
    require __DIR__ . '/footer.php';
    exit;
}

// Fetch order
$stmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id");
$stmt->execute([':id' => $id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    echo "<h1>Order not found.</h1>";
    require __DIR__ . '/footer.php';
    exit;
}

// Fetch items
$stmtItems = $pdo->prepare("SELECT * FROM order_items WHERE order_id = :id");
$stmtItems->execute([':id' => $id]);
$items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);

$allowedStatuses = ['pending', 'confirmed', 'shipped', 'completed', 'cancelled'];
$allowedPaymentStatuses = ['unpaid', 'paid_mpesa', 'paid_cash', 'refunded'];

$orderMessage = '';
$orderMessageType = 'success';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Update order status
    if ($action === 'update_status' && isset($_POST['status'])) {
        $newStatus = trim($_POST['status']);
        if (in_array($newStatus, $allowedStatuses, true)) {
            $stmtUpd = $pdo->prepare("UPDATE orders SET status = :st WHERE id = :id");
            $stmtUpd->execute([':st' => $newStatus, ':id' => $id]);
            $orderMessage = "Order status updated.";
            $orderMessageType = 'success';
            $order['status'] = $newStatus;
        } else {
            $orderMessage = "Invalid status selected.";
            $orderMessageType = 'error';
        }
    }

    // Update payment status
    if ($action === 'update_payment' && isset($_POST['payment_status'])) {
        $newPayStatus = trim($_POST['payment_status']);
        $payRef       = trim($_POST['payment_reference'] ?? '');

        if (in_array($newPayStatus, $allowedPaymentStatuses, true)) {

            // If marking as MPESA paid, require code validation
            if ($newPayStatus === 'paid_mpesa') {
                if ($payRef === '') {
                    $orderMessage = "MPESA reference is required when marking as Paid via MPESA.";
                    $orderMessageType = 'error';
                } elseif (!preg_match('/^[A-Z0-9]{8,12}$/i', $payRef)) {
                    $orderMessage = "MPESA reference seems invalid. Use 8–12 letters/numbers (e.g. QF12XYZ34).";
                    $orderMessageType = 'error';
                } else {
                    $orderMessageType = 'success';
                }
            } else {
                $orderMessageType = 'success';
            }

            if ($orderMessageType === 'success') {
                $params = [
                    ':payment_status'    => $newPayStatus,
                    ':payment_reference' => $payRef !== '' ? $payRef : null,
                    ':id'                => $id,
                ];

                $setPaidAt = '';
                if ($newPayStatus === 'paid_mpesa' || $newPayStatus === 'paid_cash') {
                    $setPaidAt = ", paid_at = IF(paid_at IS NULL, NOW(), paid_at)";
                } elseif ($newPayStatus === 'unpaid') {
                    $setPaidAt = ", paid_at = NULL, payment_reference = NULL";
                    $params[':payment_reference'] = null;
                }

                $sql = "UPDATE orders
                        SET payment_status = :payment_status, payment_reference = :payment_reference
                        $setPaidAt
                        WHERE id = :id";

                $updPay = $pdo->prepare($sql);
                $updPay->execute($params);

                if ($newPayStatus === 'paid_mpesa') {
                    $orderMessage = "Payment marked as Paid via MPESA.";
                } elseif ($newPayStatus === 'paid_cash') {
                    $orderMessage = "Payment marked as Paid (Cash).";
                } elseif ($newPayStatus === 'refunded') {
                    $orderMessage = "Payment status set to Refunded.";
                } else {
                    $orderMessage = "Payment status set to Unpaid.";
                }

                $order['payment_status']    = $newPayStatus;
                $order['payment_reference'] = $params[':payment_reference'];
            }

        } else {
            $orderMessage = "Invalid payment status selected.";
            $orderMessageType = 'error';
        }
    }
}

?>

<h1>Order #<?= (int)$order['id'] ?></h1>

<?php if ($orderMessage): ?>
    <div class="flash <?= $orderMessageType === 'error' ? 'flash-error' : 'flash-success' ?>">
        <?= htmlspecialchars($orderMessage) ?>
    </div>
<?php endif; ?>

<div class="order-detail-layout">

    <div class="product-panel">
        <h2>Customer & Delivery</h2>
        <p><strong>Name:</strong> <?= htmlspecialchars($order['customer_name']) ?></p>
        <p><strong>Phone:</strong> <?= htmlspecialchars($order['phone']) ?></p>
        <?php if ($order['email']): ?>
            <p><strong>Email:</strong> <?= htmlspecialchars($order['email']) ?></p>
        <?php endif; ?>
        <?php if ($order['town']): ?>
            <p><strong>Town:</strong> <?= htmlspecialchars($order['town']) ?></p>
        <?php endif; ?>
        <p><strong>Address:</strong><br><?= nl2br(htmlspecialchars($order['address'])) ?></p>
        <?php if ($order['notes']): ?>
            <p><strong>Notes:</strong><br><?= nl2br(htmlspecialchars($order['notes'])) ?></p>
        <?php endif; ?>
    </div>

    <div class="product-panel">
        <h2>Order Info</h2>
        <p><strong>Placed:</strong> <?= htmlspecialchars($order['created_at']) ?></p>
        <p><strong>Total:</strong> KSh <?= number_format($order['total_amount']) ?></p>

        <form method="post" style="margin-top:8px;">
            <input type="hidden" name="action" value="update_status">
            <label><strong>Status:</strong></label>
            <select name="status" style="margin-left:6px;">
                <?php foreach ($allowedStatuses as $st): ?>
                    <option value="<?= $st ?>" <?= $order['status'] === $st ? 'selected' : '' ?>>
                        <?= ucfirst($st) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button type="submit" class="admin-btn-secondary" style="margin-left:8px;">Update</button>
        </form>

        <hr style="margin:12px 0;">

        <form method="post">
            <input type="hidden" name="action" value="update_payment">
            <label><strong>Payment Status:</strong></label>
            <select name="payment_status" style="margin-left:6px;">
                <?php foreach ($allowedPaymentStatuses as $pst): ?>
                    <option value="<?= $pst ?>" <?= $order['payment_status'] === $pst ? 'selected' : '' ?>>
                        <?php
                        if ($pst === 'paid_mpesa') echo 'Paid (MPESA)';
                        elseif ($pst === 'paid_cash') echo 'Paid (Cash)';
                        elseif ($pst === 'refunded') echo 'Refunded';
                        else echo 'Unpaid';
                        ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <div style="margin-top:8px;">
                <label>MPESA / Reference Code</label>
                <input type="text" name="payment_reference"
                       value="<?= htmlspecialchars($order['payment_reference'] ?? '') ?>"
                       placeholder="e.g. QF12XYZ34"
                       style="width:100%;max-width:260px;">
            </div>
            <?php if ($order['paid_at']): ?>
                <p class="muted" style="font-size:12px;margin-top:4px;">
                    Paid at: <?= htmlspecialchars($order['paid_at']) ?>
                </p>
            <?php endif; ?>
            <button type="submit" class="admin-btn" style="margin-top:10px;">Update Payment</button>
        </form>
    </div>

</div>

<div class="product-panel" style="margin-top:20px;">
    <h2>Items</h2>

    <?php if (empty($items)): ?>
        <p class="muted">No items found for this order.</p>
    <?php else: ?>
        <table class="admin-table">
            <thead>
                <tr>
                    <th>Product</th>
                    <th>Unit Price</th>
                    <th>Qty</th>
                    <th>Line Total</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($items as $it): ?>
                    <tr>
                        <td><?= htmlspecialchars($it['product_name']) ?></td>
                        <td>KSh <?= number_format($it['unit_price']) ?></td>
                        <td><?= (int)$it['quantity'] ?></td>
                        <td>KSh <?= number_format($it['line_total']) ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<p style="margin-top:10px;">
    <a href="/admin/orders_list.php" class="admin-btn-secondary" style="text-decoration:none;">← Back to Orders</a>
</p>

<?php require __DIR__ . '/footer.php'; ?>
