<?php
// admin/index.php
$pageTitle = "Dashboard – CompNet Admin";
require __DIR__ . '/../db.php';
require __DIR__ . '/header.php';

// Today (server date)
$today = date('Y-m-d');

// 1) Total orders today
$stmtTodayOrders = $pdo->prepare("
    SELECT COUNT(*) AS cnt
    FROM orders
    WHERE DATE(created_at) = :today
");
$stmtTodayOrders->execute([':today' => $today]);
$todayOrders = (int)$stmtTodayOrders->fetchColumn();

// 2) Paid today (MPESA + cash) based on paid_at
$stmtPaidToday = $pdo->prepare("
    SELECT COALESCE(SUM(total_amount), 0) AS total
    FROM orders
    WHERE payment_status IN ('paid_mpesa', 'paid_cash')
      AND paid_at IS NOT NULL
      AND DATE(paid_at) = :today
");
$stmtPaidToday->execute([':today' => $today]);
$paidToday = (float)$stmtPaidToday->fetchColumn();

// 3) Unpaid orders count
$stmtUnpaid = $pdo->query("
    SELECT COUNT(*) AS cnt
    FROM orders
    WHERE payment_status = 'unpaid'
      AND status <> 'cancelled'
");
$unpaidOrders = (int)$stmtUnpaid->fetchColumn();

// 4) Total revenue (paid only)
$stmtTotalRevenue = $pdo->query("
    SELECT COALESCE(SUM(total_amount), 0) AS total
    FROM orders
    WHERE payment_status IN ('paid_mpesa', 'paid_cash')
");
$totalRevenue = (float)$stmtTotalRevenue->fetchColumn();

// 5) Recent 5 orders
$stmtRecent = $pdo->query("
    SELECT *
    FROM orders
    ORDER BY created_at DESC
    LIMIT 5
");
$recentOrders = $stmtRecent->fetchAll(PDO::FETCH_ASSOC);
?>

<h1>Dashboard</h1>
<p class="muted" style="margin-bottom:16px;">
    Quick overview of today’s activity and recent orders.
</p>

<div class="admin-stats-grid">
    <div class="product-panel admin-stat-card">
        <div class="admin-stat-label">Orders Today</div>
        <div class="admin-stat-value"><?= $todayOrders ?></div>
        <div class="admin-stat-sub"><?= htmlspecialchars($today) ?></div>
    </div>

    <div class="product-panel admin-stat-card">
        <div class="admin-stat-label">Paid Today (MPESA + Cash)</div>
        <div class="admin-stat-value">KSh <?= number_format($paidToday) ?></div>
        <div class="admin-stat-sub">Based on paid_at date</div>
    </div>

    <div class="product-panel admin-stat-card">
        <div class="admin-stat-label">Unpaid Orders</div>
        <div class="admin-stat-value"><?= $unpaidOrders ?></div>
        <div class="admin-stat-sub">Status ≠ cancelled</div>
    </div>

    <div class="product-panel admin-stat-card">
        <div class="admin-stat-label">Total Revenue (Paid)</div>
        <div class="admin-stat-value">KSh <?= number_format($totalRevenue) ?></div>
        <div class="admin-stat-sub">All time, paid MPESA + cash</div>
    </div>
</div>

<div class="product-panel" style="margin-top:20px;">
    <h2 style="font-size:16px;margin-bottom:8px;">Recent Orders</h2>

    <?php if (empty($recentOrders)): ?>
        <p class="muted">No orders yet.</p>
    <?php else: ?>
        <table class="admin-table">
            <thead>
                <tr>
                    <th>Order</th>
                    <th>Customer</th>
                    <th>Phone</th>
                    <th>Date</th>
                    <th>Total</th>
                    <th>Order Status</th>
                    <th>Payment</th>
                    <th></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recentOrders as $o): ?>
                    <tr>
                        <td>#<?= (int)$o['id'] ?></td>
                        <td><?= htmlspecialchars($o['customer_name']) ?></td>
                        <td><?= htmlspecialchars($o['phone']) ?></td>
                        <td><?= htmlspecialchars($o['created_at']) ?></td>
                        <td>KSh <?= number_format($o['total_amount']) ?></td>
                        <td>
                            <span class="pill pill-muted">
                                <?= htmlspecialchars(ucfirst($o['status'])) ?>
                            </span>
                        </td>
                        <td>
                            <?php
                            $payClass = 'pill-muted';
                            $payLabel = 'Unpaid';

                            if ($o['payment_status'] === 'paid_mpesa') {
                                $payClass = 'pill-success';
                                $payLabel = 'Paid (MPESA)';
                            } elseif ($o['payment_status'] === 'paid_cash') {
                                $payClass = 'pill-success';
                                $payLabel = 'Paid (Cash)';
                            } elseif ($o['payment_status'] === 'refunded') {
                                $payClass = 'pill-danger';
                                $payLabel = 'Refunded';
                            } elseif ($o['payment_status'] === 'unpaid') {
                                $payClass = 'pill-warning';
                                $payLabel = 'Unpaid';
                            }
                            ?>
                            <span class="pill <?= $payClass ?>">
                                <?= htmlspecialchars($payLabel) ?>
                            </span>
                        </td>
                        <td>
                            <a href="/admin/order_view.php?id=<?= $o['id'] ?>"
                               style="color:#0070c9;font-size:12px;text-decoration:none;">
                                View
                            </a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<?php require __DIR__ . '/footer.php'; ?>
