<?php
// admin/add_product.php
$pageTitle = "Add Product – CompNet Admin";
require __DIR__ . '/../db.php';
require __DIR__ . '/header.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Fetch categories
$categoriesStmt = $pdo->query("SELECT id, name FROM categories ORDER BY name ASC");
$categories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);

$errors = [];
$success = false;

// Simple slugify helper
function slugify(string $text): string {
    $text = strtolower(trim($text));
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = trim($text, '-');
    if (empty($text)) {
        return 'product-' . time();
    }
    return $text;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name        = trim($_POST['name'] ?? '');
    $category_id = (int)($_POST['category_id'] ?? 0);
    $price       = (float)($_POST['price'] ?? 0);
    $old_price   = $_POST['old_price'] !== '' ? (float)$_POST['old_price'] : null;
    $short_meta  = trim($_POST['short_meta'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $main_image  = trim($_POST['main_image'] ?? '');
    $tags        = trim($_POST['tags'] ?? '');
    $condition   = trim($_POST['condition'] ?? '');
    $is_featured = !empty($_POST['is_featured']) ? 1 : 0;
    $is_active   = !empty($_POST['is_active']) ? 1 : 0;

    if ($name === '') {
        $errors[] = "Product name is required.";
    }
    if ($category_id <= 0) {
        $errors[] = "Category is required.";
    }
    if ($price <= 0) {
        $errors[] = "Price must be greater than zero.";
    }

    if (empty($errors)) {
        $slug = slugify($name);

        $stmt = $pdo->prepare("
            INSERT INTO products
                (name, slug, category_id, price, old_price, short_meta, description, main_image, tags, condition, is_featured, is_active)
            VALUES
                (:name, :slug, :category_id, :price, :old_price, :short_meta, :description, :main_image, :tags, :condition, :is_featured, :is_active)
        ");

        $stmt->execute([
            ':name'        => $name,
            ':slug'        => $slug,
            ':category_id' => $category_id,
            ':price'       => $price,
            ':old_price'   => $old_price,
            ':short_meta'  => $short_meta ?: null,
            ':description' => $description ?: null,
            ':main_image'  => $main_image ?: null,
            ':tags'        => $tags ?: null,
            ':condition'   => $condition ?: null,
            ':is_featured' => $is_featured,
            ':is_active'   => $is_active,
        ]);

        $_SESSION['flash'] = [
            'type'    => 'success',
            'message' => 'Product created successfully.',
        ];

        header("Location: /admin/products_list.php");
        exit;
    }
}
?>

<h1>Add Product</h1>

<?php if (!empty($errors)): ?>
    <div class="flash flash-error">
        <?= htmlspecialchars(implode(' ', $errors)) ?>
    </div>
<?php endif; ?>

<div class="product-panel">
    <form method="post" style="display:grid;grid-template-columns:1fr 1fr;gap:12px;">
        <div style="grid-column:1 / -1;">
            <label>Name *</label>
            <input type="text" name="name" required
                   value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
        </div>

        <div>
            <label>Category *</label>
            <select name="category_id" required>
                <option value="">Select category</option>
                <?php foreach ($categories as $c): ?>
                    <option value="<?= $c['id'] ?>" <?= (($_POST['category_id'] ?? '') == $c['id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($c['name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div>
            <label>Condition</label>
            <select name="condition">
                <option value="">—</option>
                <?php
                $cond = $_POST['condition'] ?? '';
                ?>
                <option value="new"    <?= $cond === 'new' ? 'selected' : '' ?>>New</option>
                <option value="ex-uk"  <?= $cond === 'ex-uk' ? 'selected' : '' ?>>Ex-UK</option>
                <option value="refurb" <?= $cond === 'refurb' ? 'selected' : '' ?>>Refurbished</option>
                <option value="used"   <?= $cond === 'used' ? 'selected' : '' ?>>Used</option>
            </select>
        </div>

        <div>
            <label>Price (KSh) *</label>
            <input type="number" step="0.01" name="price" required
                   value="<?= htmlspecialchars($_POST['price'] ?? '') ?>">
        </div>

        <div>
            <label>Old Price (KSh)</label>
            <input type="number" step="0.01" name="old_price"
                   value="<?= htmlspecialchars($_POST['old_price'] ?? '') ?>">
        </div>

        <div>
            <label>Main Image URL</label>
            <input type="text" name="main_image"
                   value="<?= htmlspecialchars($_POST['main_image'] ?? '') ?>">
        </div>

        <div style="grid-column:1 / -1;">
            <label>Short Meta (summary)</label>
            <input type="text" name="short_meta"
                   value="<?= htmlspecialchars($_POST['short_meta'] ?? '') ?>">
        </div>

        <div style="grid-column:1 / -1;">
            <label>Tags (comma-separated)</label>
            <input type="text" name="tags"
                   value="<?= htmlspecialchars($_POST['tags'] ?? '') ?>">
        </div>

        <div style="grid-column:1 / -1;">
            <label>Description</label>
            <textarea name="description" rows="5"><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>
        </div>

        <div>
            <label>
                <input type="checkbox" name="is_featured" value="1"
                       <?= !empty($_POST['is_featured']) ? 'checked' : '' ?>>
                Featured
            </label>
        </div>

        <div>
            <label>
                <input type="checkbox" name="is_active" value="1"
                       <?= isset($_POST['is_active']) ? ( $_POST['is_active'] ? 'checked' : '' ) : 'checked' ?>>
                Active
            </label>
        </div>

        <div style="grid-column:1 / -1;display:flex;justify-content:space-between;align-items:center;margin-top:10px;">
            <a href="/admin/products_list.php" class="admin-btn-secondary" style="text-decoration:none;">← Back</a>
            <button type="submit" class="admin-btn">Save Product</button>
        </div>
    </form>
</div>

<?php require __DIR__ . '/footer.php'; ?>
